/************************************************************/
/*                                                          */
/* Project name: Codes over Z4 in MAGMA                     */
/* Test file name: testCodesOverZ4_part2.m                  */
/*                                                          */
/* Comments: Black-box tests for the functions              */
/*           StandardFormInverted, MinRowsGeneratorMatrix   */
/*           StandardFormDual, StandardFormDualInfo, DualZ4 */
/*           MinRowsParityCheckMatrix, SpanZ2CodeZ4,        */
/*           DimensionOfSpanZ2, KernelZ2CodeZ4, and         */
/*           DimensionOfKernelZ2                            */
/*           included in the CodesOverZ4.m file             */
/*                                                          */
/* Authors: M. Villanueva and J. Pujol                      */
/*                                                          */
/* Revision version and last date: v1.0   2016/07/15        */
/*                                 v1.1   2016/08/13        */
/*                                 v1.2   2016/09/12        */
/*                                                          */
/************************************************************/
 
SetAssertions(true);
Alarm(30*60);

Z4 := Integers(4);

/****************************************************************/
/*                                                              */  
/* Function name: StandardFormInverted                          */
/* Parameters: C                                                */
/* Function description: Given a code C over Z4, return a       */
/*   permutation-equivalent code S in standard form, together   */
/*   with the corresponding isomorphism from C onto S. The      */
/*   code S is generated by a matrix in standard form having    */
/*   the rows and columns in inverted order compared to the one */
/*   given by the function StandardFormInfo(C), that is, the    */
/*   matrix [2C' 2Ik2  0 ] instead of the matrix [Ik1  A    B ] */ 
/*          [ B'  A'  Ik1]                       [0   2Ik2  2C],*/
/*   Ik1 and Ik2 are the k1 x k1 and k2 x k2 identity matrices, */
/*   respectively, A, A', C and C' are Z2-matrices, and B and B'*/
/*   are Z4-matrices.                                           */               
/* Input parameters description:                                */
/*   - C : Code over Z4                                         */
/* Output parameters description:                               */
/*   - Code over Z4 in standard form                            */
/*   - Isomorphism from C onto S                                */
/*                                                              */
/* Signature: (<CodeLinRng> C) -> CodeLinRng, Map               */
/*                                                              */
/****************************************************************/
/****************************************************************/
/*                                                              */  
/* Function name: StandardFormDual                              */
/* Parameters: C                                                */
/* Function description: Given a code C over Z4 of lenght n,    */
/*   return the dual of a permutation-equivalent code S in      */
/*   standard form, together with the corresponding isomorphism */
/*   from the dual of C onto the dual of S. Since S is generated*/
/*   by a matrix of the form [[Ik1, A, B], [0, 2Ik2, 2C]], the  */
/*   dual of S is generated by the matrix                       */
/*   [[-(AC+B)^t, C^t, I_(n-k1-k2)], [2A^t, 2Ik2, 0]], where    */
/*   Ik1 and Ik2 are the k1 x k1 and k2 x k2 identity matrices, */
/*   respectively, A and C are Z2-matrices, and B is a Z4-matrix*/                     
/* Input parameters description:                                */
/*   - C : Code over Z4                                         */
/* Output parameters description:                               */
/*   - Dual code of S over Z4 in standard form                  */
/*   - Isomorphism from dual of C onto dual of S                */
/*                                                              */
/* Signature: (<CodeLinRng> C) -> CodeLinRng, Map               */
/*                                                              */
/****************************************************************/
/****************************************************************/
/*                                                              */  
/* Function name: MinRowsParityCheckMatrix                      */
/* Parameters: C                                                */
/* Function description: A parity check matrix for the code C   */
/*   over Z4 of lenght n and type 2^gamma 4^delta, with the     */
/*   minimum number of rows, that is, with gamma rows of order  */
/*   two and n-gamma-delta rows of order four. This function    */
/*   should be faster for most codes over Z4 than the general   */
/*   function ParityCheckMatrix(C) for codes over finite rings. */
/*   Another parity check matrix for the code C can be obtained */
/*   as the generator matrix of the dual of C with the minimum  */
/*   number of rows, that is, as                                */
/*   MinRowsGeneratorMatrix(DualZ4(C)).                         */
/* Input parameters description:                                */
/*   - C : Code over Z4                                         */
/* Output parameters description:                               */
/*   - Generator matrix of C over Z4                            */
/*                                                              */
/* Signature: (<CodeLinRng> C) -> ModMatRngElt                  */
/*                                                              */
/****************************************************************/
/****************************************************************/
/*                                                              */  
/* Function name: DualZ4                                        */
/* Parameters: C                                                */
/* Function description: The dual D of the code C over Z4 of    */
/*   length n. The dual consists of all codewords in the        */
/*   Z4-space V=Z4^n which are orthogonal to all codewords of C.*/
/*   This function should be faster for most codes over Z4 than */
/*   the general function Dual(C) for codes over finite rings.  */
/* Input parameters description:                                */
/*   - C : Code over Z4                                         */
/* Output parameters description:                               */
/*   - Code D over Z4                                           */
/*                                                              */
/* Signature: (<CodeLinRng> C) -> CodeLinRng                    */
/*                                                              */
/****************************************************************/
print "test 1: Trivial zero code over Z4
               #C = 1, length = 5";
C := ZeroCode(Z4, 5);
n := Length(C);
delta, gamma := Z4Type(C);
deltaDual := n-gamma-delta;

expectedOutputDualC := UniverseCode(Z4, 5);
expectedOutputDualCs := UniverseCode(Z4, 5);

expectedSpanCodeZ4 := C;
_, expectedSpanCodeZ2 := HasLinearGrayMapImage(expectedSpanCodeZ4);
expectedKernelCodeZ4 := C;
_, expectedKernelCodeZ2 := HasLinearGrayMapImage(expectedKernelCodeZ4);

// Test StandardFormInverted, MinRowsGeneratorMatrix functions
OutputMatrixGs, p := StandardFormInfoInverted(C);
OutputCs, f := StandardFormInverted(C);
OutputMatrixG := MinRowsGeneratorMatrix(C); 
assert Submatrix(OutputMatrixGs, gamma+1, n-delta+1, delta, delta) eq IdentityMatrix(Z4,delta);
assert Submatrix(OutputMatrixGs, 1, n-delta-gamma+1, gamma, gamma) eq 2*IdentityMatrix(Z4,gamma); 
assert OutputCs eq LinearCode(OutputMatrixGs);
assert OutputCs eq C^p;
//not checked because it is an empty matrix 
//assert OutputCs eq LinearCode(Matrix([f(OutputMatrixG[i]) : i in [1..gamma+delta]]));
assert C eq LinearCode(OutputMatrixG);
assert Nrows(OutputMatrixG) eq gamma+delta;
assert 2*RowSubmatrix(OutputMatrixG, gamma) eq ZeroMatrix(Z4, gamma, n);

// Test StandardFormDual, DualZ4 and MinRowsParityCheckMatrix functions
OutputMatrixHs, p := StandardFormDualInfo(C);
OutputDualCs, f := StandardFormDual(C);
OutputMatrixH := MinRowsParityCheckMatrix(C);
OutputDualC := DualZ4(C);
assert Submatrix(OutputMatrixHs, 1, delta+gamma+1, deltaDual, deltaDual) eq IdentityMatrix(Z4, deltaDual);
assert Submatrix(OutputMatrixHs, deltaDual+1, delta+1, gamma, gamma) eq 2*IdentityMatrix(Z4, gamma); 
assert OutputMatrixHs eq OutputMatrixH^p;
assert OutputMatrixHs eq Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]); 
assert OutputDualCs eq LinearCode(OutputMatrixHs);
assert OutputDualCs eq expectedOutputDualC^p; 
assert OutputDualCs eq LinearCode(Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]));
assert OutputDualC eq LinearCode(OutputMatrixH);
assert Nrows(OutputMatrixH) eq gamma+deltaDual;
assert 2*RowSubmatrix(OutputMatrixH, gamma) eq ZeroMatrix(Z4, gamma, n);
assert OutputDualC eq expectedOutputDualC;
assert OutputDualCs eq expectedOutputDualCs;

// Test SpanZ2CodeZ4, DimensionOfSpanZ2, KernelZ2CodeZ4, DimensionOfKernelZ2
OutputSpanCodeZ4, OutputSpanCodeZ2 := SpanZ2CodeZ4(C);
OutputKernelCodeZ4, OutputKernelCodeZ2 := KernelZ2CodeZ4(C);
OutputDimensionSpan := DimensionOfSpanZ2(C);
OutputDimensionKernel := DimensionOfKernelZ2(C);
assert OutputSpanCodeZ4 eq expectedSpanCodeZ4;
assert OutputSpanCodeZ2 eq expectedSpanCodeZ2;
assert OutputKernelCodeZ4 eq expectedKernelCodeZ4;
assert OutputKernelCodeZ2 eq expectedKernelCodeZ2;
assert OutputDimensionSpan eq Dimension(expectedSpanCodeZ2);
assert OutputDimensionKernel eq Dimension(expectedKernelCodeZ2);

/****************************************************************/
print "test 2: Trivial universe code over Z4
               #C = 4096, lenght = 6";
C := UniverseCode(Z4, 6);
n := Length(C);
delta, gamma := Z4Type(C);
deltaDual := n-gamma-delta;

expectedOutputDualC := ZeroCode(Z4, 6);
expectedOutputDualCs := ZeroCode(Z4, 6);

expectedSpanCodeZ4 := UniverseCode(Z4, 6);
_, expectedSpanCodeZ2 := HasLinearGrayMapImage(expectedSpanCodeZ4);
expectedKernelCodeZ4 := UniverseCode(Z4, 6);
_, expectedKernelCodeZ2 := HasLinearGrayMapImage(expectedKernelCodeZ4);

// Test StandardFormInverted, MinRowsGeneratorMatrix functions
OutputMatrixGs, p := StandardFormInfoInverted(C);
OutputCs, f := StandardFormInverted(C);
OutputMatrixG := MinRowsGeneratorMatrix(C); 
assert Submatrix(OutputMatrixGs, gamma+1, n-delta+1, delta, delta) eq IdentityMatrix(Z4,delta);
assert Submatrix(OutputMatrixGs, 1, n-delta-gamma+1, gamma, gamma) eq 2*IdentityMatrix(Z4,gamma); 
assert OutputCs eq LinearCode(OutputMatrixGs);
assert OutputCs eq C^p;
assert OutputCs eq LinearCode(Matrix([f(OutputMatrixG[i]) : i in [1..gamma+delta]]));
assert C eq LinearCode(OutputMatrixG);
assert Nrows(OutputMatrixG) eq gamma+delta;
assert 2*RowSubmatrix(OutputMatrixG, gamma) eq ZeroMatrix(Z4, gamma, n);

// Test StandardFormDual, DualZ4 and MinRowsParityCheckMatrix functions
OutputMatrixHs, p := StandardFormDualInfo(C);
OutputDualCs, f := StandardFormDual(C);
OutputMatrixH := MinRowsParityCheckMatrix(C);
OutputDualC := DualZ4(C);
assert Submatrix(OutputMatrixHs, 1, delta+gamma+1, deltaDual, deltaDual) eq IdentityMatrix(Z4, deltaDual);
assert Submatrix(OutputMatrixHs, deltaDual+1, delta+1, gamma, gamma) eq 2*IdentityMatrix(Z4, gamma); 
assert OutputMatrixHs eq OutputMatrixH^p;
//not checked because it is an empty matrix 
//assert OutputMatrixHs eq Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]); 
assert OutputDualCs eq LinearCode(OutputMatrixHs);
assert OutputDualCs eq expectedOutputDualC^p; 
//not checked because it is an empty matrix 
//assert OutputDualCs eq LinearCode(Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]));
assert OutputDualC eq LinearCode(OutputMatrixH);
assert Nrows(OutputMatrixH) eq gamma+deltaDual;
assert 2*RowSubmatrix(OutputMatrixH, gamma) eq ZeroMatrix(Z4, gamma, n);
assert OutputDualC eq expectedOutputDualC;
assert OutputDualCs eq expectedOutputDualCs;

// Test SpanZ2CodeZ4, DimensionOfSpanZ2, KernelZ2CodeZ4, DimensionOfKernelZ2
OutputSpanCodeZ4, OutputSpanCodeZ2 := SpanZ2CodeZ4(C);
OutputKernelCodeZ4, OutputKernelCodeZ2 := KernelZ2CodeZ4(C);
OutputDimensionSpan := DimensionOfSpanZ2(C);
OutputDimensionKernel := DimensionOfKernelZ2(C);
assert OutputSpanCodeZ4 eq expectedSpanCodeZ4;
assert OutputSpanCodeZ2 eq expectedSpanCodeZ2;
assert OutputKernelCodeZ4 eq expectedKernelCodeZ4;
assert OutputKernelCodeZ2 eq expectedKernelCodeZ2;
assert OutputDimensionSpan eq Dimension(expectedSpanCodeZ2);
assert OutputDimensionKernel eq Dimension(expectedKernelCodeZ2);

/****************************************************************/
print "test 3: Extended Hamming code over Z4
               #C = 2048, length = 8, and type 4^5 2^1";
C := ExtendedPerfectCodeZ4(2, 4);
n := Length(C);
delta, gamma := Z4Type(C);
deltaDual := n-gamma-delta;

matrixHs := Matrix(Z4, [[3,2,3,1,2,0,1,0],
                        [0,1,2,2,1,1,0,1],
                        [2,2,2,0,0,2,0,0]]);
expectedOutputDualCs := LinearCode(matrixHs);
expectedOutputDualC := Dual(C);

expectedSpanCodeZ4 := LinearCode(Matrix(Z4, [[1,0,0,1,0,0,1,1],
                                             [0,1,0,1,0,0,0,0],
                                             [0,0,1,1,0,0,1,1],
                                             [0,0,0,2,0,0,0,0],
                                             [0,0,0,0,1,0,1,0],
                                             [0,0,0,0,0,1,0,1],
                                             [0,0,0,0,0,0,2,0],
                                             [0,0,0,0,0,0,0,2]]));
_, expectedSpanCodeZ2 := HasLinearGrayMapImage(expectedSpanCodeZ4);
                                             
expectedKernelCodeZ4 := LinearCode(Matrix(Z4, [[1,0,1,0,1,0,1,0],
                                               [0,1,0,1,0,1,0,1],
                                               [0,0,2,0,0,0,2,0],
                                               [0,0,0,2,0,0,0,2],
                                               [0,0,0,0,2,0,2,0],
                                               [0,0,0,0,0,2,0,2]]));
_, expectedKernelCodeZ2 := HasLinearGrayMapImage(expectedKernelCodeZ4);

// Test StandardFormInverted, MinRowsGeneratorMatrix functions
OutputMatrixGs, p := StandardFormInfoInverted(C);
OutputCs, f := StandardFormInverted(C);
OutputMatrixG := MinRowsGeneratorMatrix(C); 
assert Submatrix(OutputMatrixGs, gamma+1, n-delta+1, delta, delta) eq IdentityMatrix(Z4,delta);
assert Submatrix(OutputMatrixGs, 1, n-delta-gamma+1, gamma, gamma) eq 2*IdentityMatrix(Z4,gamma); 
assert OutputCs eq LinearCode(OutputMatrixGs);
assert OutputCs eq C^p;
assert OutputCs eq LinearCode(Matrix([f(OutputMatrixG[i]) : i in [1..gamma+delta]]));
assert C eq LinearCode(OutputMatrixG);
assert Nrows(OutputMatrixG) eq gamma+delta;
assert 2*RowSubmatrix(OutputMatrixG, gamma) eq ZeroMatrix(Z4, gamma, n);

// Test StandardFormDual, DualZ4 and MinRowsParityCheckMatrix functions
OutputMatrixHs, p := StandardFormDualInfo(C);
OutputDualCs, f := StandardFormDual(C);
OutputMatrixH := MinRowsParityCheckMatrix(C);
OutputDualC := DualZ4(C);
assert Submatrix(OutputMatrixHs, 1, delta+gamma+1, deltaDual, deltaDual) eq IdentityMatrix(Z4, deltaDual);
assert Submatrix(OutputMatrixHs, deltaDual+1, delta+1, gamma, gamma) eq 2*IdentityMatrix(Z4, gamma); 
assert OutputDualCs eq LinearCode(OutputMatrixHs);
assert OutputDualCs eq expectedOutputDualC^p; 
assert OutputDualCs eq LinearCode(Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]));
assert OutputDualC eq LinearCode(OutputMatrixH);
assert Nrows(OutputMatrixH) eq gamma+deltaDual;
assert 2*RowSubmatrix(OutputMatrixH, gamma) eq ZeroMatrix(Z4, gamma, n);
assert OutputDualC eq expectedOutputDualC;
assert OutputDualCs eq expectedOutputDualCs;

// Test SpanZ2CodeZ4, DimensionOfSpanZ2, KernelZ2CodeZ4, DimensionOfKernelZ2
OutputSpanCodeZ4, OutputSpanCodeZ2 := SpanZ2CodeZ4(C);
OutputKernelCodeZ4, OutputKernelCodeZ2 := KernelZ2CodeZ4(C);
OutputDimensionSpan := DimensionOfSpanZ2(C);
OutputDimensionKernel := DimensionOfKernelZ2(C);
assert OutputSpanCodeZ4 eq expectedSpanCodeZ4;
assert OutputSpanCodeZ2 eq expectedSpanCodeZ2;
assert OutputKernelCodeZ4 eq expectedKernelCodeZ4;
assert OutputKernelCodeZ2 eq expectedKernelCodeZ2;
assert OutputDimensionSpan eq Dimension(expectedSpanCodeZ2);
assert OutputDimensionKernel eq Dimension(expectedKernelCodeZ2);

/****************************************************************/
print "test 4: Hadamard code over Z4
               #C = 64, length = 16, and type 4^3 2^0";
C := HadamardCodeZ4(3, 5);
n := Length(C);
delta, gamma := Z4Type(C);
deltaDual := n-gamma-delta;

matrixHs := Matrix(Z4, [[2,1,0,1,0,0,0,0,0,0,0,0,0,0,0,0],
                        [1,2,0,0,1,0,0,0,0,0,0,0,0,0,0,0],
                        [1,3,3,0,0,1,0,0,0,0,0,0,0,0,0,0],
                        [2,2,3,0,0,0,1,0,0,0,0,0,0,0,0,0],
                        [3,1,3,0,0,0,0,1,0,0,0,0,0,0,0,0],
                        [1,0,2,0,0,0,0,0,1,0,0,0,0,0,0,0],
                        [2,3,2,0,0,0,0,0,0,1,0,0,0,0,0,0],
                        [3,2,2,0,0,0,0,0,0,0,1,0,0,0,0,0],
                        [0,1,2,0,0,0,0,0,0,0,0,1,0,0,0,0],
                        [2,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0],
                        [3,3,1,0,0,0,0,0,0,0,0,0,0,1,0,0],
                        [0,2,1,0,0,0,0,0,0,0,0,0,0,0,1,0],
                        [1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,1]]);
expectedOutputDualCs := LinearCode(matrixHs);
expectedOutputDualC := Dual(C);

expectedSpanCodeZ4 := LinearCode(Matrix(Z4,[[1,0,3,2,0,1,2,3,3,2,1,0,2,3,0,1],
                                            [0,1,2,3,0,1,2,3,0,1,2,3,0,1,2,3],
                                            [0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3],
                                            [0,0,0,0,0,2,0,2,0,0,0,0,0,2,0,2]]));
_,expectedSpanCodeZ2 := HasLinearGrayMapImage(expectedSpanCodeZ4);
                                             
expectedKernelCodeZ4 := LinearCode(Matrix(Z4,[[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],
                                              [0,2,0,2,0,2,0,2,0,2,0,2,0,2,0,2],
                                              [0,0,0,0,2,2,2,2,0,0,0,0,2,2,2,2]]));
_,expectedKernelCodeZ2 := HasLinearGrayMapImage(expectedKernelCodeZ4);

// Test StandardFormInverted, MinRowsGeneratorMatrix functions
OutputMatrixGs, p := StandardFormInfoInverted(C);
OutputCs, f := StandardFormInverted(C);
OutputMatrixG := MinRowsGeneratorMatrix(C); 
assert Submatrix(OutputMatrixGs, gamma+1, n-delta+1, delta, delta) eq IdentityMatrix(Z4,delta);
assert Submatrix(OutputMatrixGs, 1, n-delta-gamma+1, gamma, gamma) eq 2*IdentityMatrix(Z4,gamma); 
assert OutputCs eq LinearCode(OutputMatrixGs);
assert OutputCs eq C^p;
assert OutputCs eq LinearCode(Matrix([f(OutputMatrixG[i]) : i in [1..gamma+delta]]));
assert C eq LinearCode(OutputMatrixG);
assert Nrows(OutputMatrixG) eq gamma+delta;
assert 2*RowSubmatrix(OutputMatrixG, gamma) eq ZeroMatrix(Z4, gamma, n);

// Test StandardFormDual, DualZ4 and MinRowsParityCheckMatrix functions
OutputMatrixHs, p := StandardFormDualInfo(C);
OutputDualCs, f := StandardFormDual(C);
OutputMatrixH := MinRowsParityCheckMatrix(C);
OutputDualC := DualZ4(C);
assert Submatrix(OutputMatrixHs, 1, delta+gamma+1, deltaDual, deltaDual) eq IdentityMatrix(Z4, deltaDual);
assert Submatrix(OutputMatrixHs, deltaDual+1, delta+1, gamma, gamma) eq 2*IdentityMatrix(Z4, gamma); 
assert OutputMatrixHs eq OutputMatrixH^p;
assert OutputMatrixHs eq Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]); 
assert OutputDualCs eq LinearCode(OutputMatrixHs);
assert OutputDualCs eq expectedOutputDualC^p; 
assert OutputDualCs eq LinearCode(Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]));
assert OutputDualC eq LinearCode(OutputMatrixH);
assert Nrows(OutputMatrixH) eq gamma+deltaDual;
assert 2*RowSubmatrix(OutputMatrixH, gamma) eq ZeroMatrix(Z4, gamma, n);
assert OutputDualC eq expectedOutputDualC;
assert OutputDualCs eq expectedOutputDualCs;

// Test SpanZ2CodeZ4, DimensionOfSpanZ2, KernelZ2CodeZ4, DimensionOfKernelZ2
OutputSpanCodeZ4, OutputSpanCodeZ2 := SpanZ2CodeZ4(C);
OutputKernelCodeZ4, OutputKernelCodeZ2 := KernelZ2CodeZ4(C);
OutputDimensionSpan := DimensionOfSpanZ2(C);
OutputDimensionKernel := DimensionOfKernelZ2(C);
assert OutputSpanCodeZ4 eq expectedSpanCodeZ4;
assert OutputSpanCodeZ2 eq expectedSpanCodeZ2;
assert OutputKernelCodeZ4 eq expectedKernelCodeZ4;
assert OutputKernelCodeZ2 eq expectedKernelCodeZ2;
assert OutputDimensionSpan eq Dimension(expectedSpanCodeZ2);
assert OutputDimensionKernel eq Dimension(expectedKernelCodeZ2);

/****************************************************************/
print "test 5: Preparata code
               #C = 256,length = 8,and type 4^4 2^0";
C := PreparataCode(3);
n := Length(C);
delta, gamma := Z4Type(C);
deltaDual := n-gamma-delta;

matrixHs := Matrix(Z4, [[1,2,3,1,1,0,0,0],
                        [3,3,3,2,0,1,0,0],
                        [2,3,1,1,0,0,1,0],
                        [3,1,2,1,0,0,0,1]]);
expectedOutputDualCs := LinearCode(matrixHs);
expectedOutputDualC := Dual(C);

expectedSpanCodeZ4 := LinearCode(Matrix(Z4, [[1,0,0,0,1,1,0,1],
                                             [0,1,0,0,0,1,1,1],
                                             [0,0,1,0,1,1,1,0],
                                             [0,0,0,1,1,0,1,1],
                                             [0,0,0,0,2,0,0,2],
                                             [0,0,0,0,0,2,0,2],
                                             [0,0,0,0,0,0,2,2]]));
_,expectedSpanCodeZ2 := HasLinearGrayMapImage(expectedSpanCodeZ4);
                                             
expectedKernelCodeZ4 := LinearCode(Matrix(Z4, [[1,1,1,1,1,1,1,1],
                                               [0,2,0,0,0,2,2,2],
                                               [0,0,2,0,2,2,2,0],
                                               [0,0,0,2,2,0,2,2]]));
_,expectedKernelCodeZ2 := HasLinearGrayMapImage(expectedKernelCodeZ4);

// Test StandardFormInverted, MinRowsGeneratorMatrix functions
OutputMatrixGs, p := StandardFormInfoInverted(C);
OutputCs, f := StandardFormInverted(C);
OutputMatrixG := MinRowsGeneratorMatrix(C); 
assert Submatrix(OutputMatrixGs, gamma+1, n-delta+1, delta, delta) eq IdentityMatrix(Z4,delta);
assert Submatrix(OutputMatrixGs, 1, n-delta-gamma+1, gamma, gamma) eq 2*IdentityMatrix(Z4,gamma); 
assert OutputCs eq LinearCode(OutputMatrixGs);
assert OutputCs eq C^p;
assert OutputCs eq LinearCode(Matrix([f(OutputMatrixG[i]) : i in [1..gamma+delta]]));
assert C eq LinearCode(OutputMatrixG);
assert Nrows(OutputMatrixG) eq gamma+delta;
assert 2*RowSubmatrix(OutputMatrixG, gamma) eq ZeroMatrix(Z4, gamma, n);

// Test StandardFormDual, DualZ4 and MinRowsParityCheckMatrix functions
OutputMatrixHs, p := StandardFormDualInfo(C);
OutputDualCs, f := StandardFormDual(C);
OutputMatrixH := MinRowsParityCheckMatrix(C);
OutputDualC := DualZ4(C);
assert Submatrix(OutputMatrixHs, 1, delta+gamma+1, deltaDual, deltaDual) eq IdentityMatrix(Z4, deltaDual);
assert Submatrix(OutputMatrixHs, deltaDual+1, delta+1, gamma, gamma) eq 2*IdentityMatrix(Z4, gamma); 
assert OutputMatrixHs eq OutputMatrixH^p;
assert OutputMatrixHs eq Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]); 
assert OutputDualCs eq LinearCode(OutputMatrixHs);
assert OutputDualCs eq expectedOutputDualC^p; 
assert OutputDualCs eq LinearCode(Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]));
assert OutputDualC eq LinearCode(OutputMatrixH);
assert Nrows(OutputMatrixH) eq gamma+deltaDual;
assert 2*RowSubmatrix(OutputMatrixH, gamma) eq ZeroMatrix(Z4, gamma, n);
assert OutputDualC eq expectedOutputDualC;
assert OutputDualCs eq expectedOutputDualCs;

// Test SpanZ2CodeZ4, DimensionOfSpanZ2, KernelZ2CodeZ4, DimensionOfKernelZ2
OutputSpanCodeZ4, OutputSpanCodeZ2 := SpanZ2CodeZ4(C);
OutputKernelCodeZ4, OutputKernelCodeZ2 := KernelZ2CodeZ4(C);
OutputDimensionSpan := DimensionOfSpanZ2(C);
OutputDimensionKernel := DimensionOfKernelZ2(C);
assert OutputSpanCodeZ4 eq expectedSpanCodeZ4;
assert OutputSpanCodeZ2 eq expectedSpanCodeZ2;
assert OutputKernelCodeZ4 eq expectedKernelCodeZ4;
assert OutputKernelCodeZ2 eq expectedKernelCodeZ2;
assert OutputDimensionSpan eq Dimension(expectedSpanCodeZ2);
assert OutputDimensionKernel eq Dimension(expectedKernelCodeZ2);

/****************************************************************/
print "test 6: Z4-linear code
               #C = 32,length = 32,and type 4^0 2^5";
G := Matrix(Z4, [[2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2],
		         [0,2,0,0,0,2,0,0,2,2,0,2,0,2,2,2],
                 [0,0,2,0,0,2,2,0,2,0,2,2,2,2,0,0],
                 [0,0,0,2,0,0,2,2,0,2,0,2,2,2,2,0],
                 [0,0,0,0,2,0,0,2,2,0,2,0,2,2,2,2]]);
C := LinearCode(G);
n := Length(C);
delta, gamma := Z4Type(C);
deltaDual := n-gamma-delta;

matrixHs := Matrix(Z4, [[1,1,1,0,0,1,0,0,0,0,0,0,0,0,0,0],
                        [1,0,1,1,0,0,1,0,0,0,0,0,0,0,0,0],
                        [1,0,0,1,1,0,0,1,0,0,0,0,0,0,0,0],
                        [0,1,1,0,1,0,0,0,1,0,0,0,0,0,0,0],
                        [1,1,0,1,0,0,0,0,0,1,0,0,0,0,0,0],
                        [1,0,1,0,1,0,0,0,0,0,1,0,0,0,0,0],
                        [0,1,1,1,0,0,0,0,0,0,0,1,0,0,0,0],
                        [0,0,1,1,1,0,0,0,0,0,0,0,1,0,0,0],
                        [1,1,1,1,1,0,0,0,0,0,0,0,0,1,0,0],
                        [0,1,0,1,1,0,0,0,0,0,0,0,0,0,1,0],
                        [1,1,0,0,1,0,0,0,0,0,0,0,0,0,0,1],
                        [2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0],
                        [0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0],
                        [0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0],
                        [0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0],
                        [0,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0]]);
expectedOutputDualCs := LinearCode(matrixHs);
expectedOutputDualC := Dual(C);

expectedSpanCodeZ4 := C;
_,expectedSpanCodeZ2 := HasLinearGrayMapImage(expectedSpanCodeZ4);
                                             
expectedKernelCodeZ4 := C;
_,expectedKernelCodeZ2 := HasLinearGrayMapImage(expectedKernelCodeZ4);

// Test StandardFormInverted, MinRowsGeneratorMatrix functions
OutputMatrixGs, p := StandardFormInfoInverted(C);
OutputCs, f := StandardFormInverted(C);
OutputMatrixG := MinRowsGeneratorMatrix(C); 
assert Submatrix(OutputMatrixGs, gamma+1, n-delta+1, delta, delta) eq IdentityMatrix(Z4,delta);
assert Submatrix(OutputMatrixGs, 1, n-delta-gamma+1, gamma, gamma) eq 2*IdentityMatrix(Z4,gamma); 
assert OutputCs eq LinearCode(OutputMatrixGs);
assert OutputCs eq C^p;
assert OutputCs eq LinearCode(Matrix([f(OutputMatrixG[i]) : i in [1..gamma+delta]]));
assert C eq LinearCode(OutputMatrixG);
assert Nrows(OutputMatrixG) eq gamma+delta;
assert 2*RowSubmatrix(OutputMatrixG, gamma) eq ZeroMatrix(Z4, gamma, n);

// Test StandardFormDual, DualZ4 and MinRowsParityCheckMatrix functions
OutputMatrixHs, p := StandardFormDualInfo(C);
OutputDualCs, f := StandardFormDual(C);
OutputMatrixH := MinRowsParityCheckMatrix(C);
OutputDualC := DualZ4(C);
assert Submatrix(OutputMatrixHs, 1, delta+gamma+1, deltaDual, deltaDual) eq IdentityMatrix(Z4, deltaDual);
assert Submatrix(OutputMatrixHs, deltaDual+1, delta+1, gamma, gamma) eq 2*IdentityMatrix(Z4, gamma); 
assert OutputDualCs eq LinearCode(OutputMatrixHs);
assert OutputDualCs eq expectedOutputDualC^p; 
assert OutputDualCs eq LinearCode(Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]));
assert OutputDualC eq LinearCode(OutputMatrixH);
assert Nrows(OutputMatrixH) eq gamma+deltaDual;
assert 2*RowSubmatrix(OutputMatrixH, gamma) eq ZeroMatrix(Z4, gamma, n);
assert OutputDualC eq expectedOutputDualC;
assert OutputDualCs eq expectedOutputDualCs;

// Test SpanZ2CodeZ4, DimensionOfSpanZ2, KernelZ2CodeZ4, DimensionOfKernelZ2
OutputSpanCodeZ4, OutputSpanCodeZ2 := SpanZ2CodeZ4(C);
OutputKernelCodeZ4, OutputKernelCodeZ2 := KernelZ2CodeZ4(C);
OutputDimensionSpan := DimensionOfSpanZ2(C);
OutputDimensionKernel := DimensionOfKernelZ2(C);
assert OutputSpanCodeZ4 eq expectedSpanCodeZ4;
assert OutputSpanCodeZ2 eq expectedSpanCodeZ2;
assert OutputKernelCodeZ4 eq expectedKernelCodeZ4;
assert OutputKernelCodeZ2 eq expectedKernelCodeZ2;
assert OutputDimensionSpan eq Dimension(expectedSpanCodeZ2);
assert OutputDimensionKernel eq Dimension(expectedKernelCodeZ2);

/****************************************************************/
print "test 7: Z4-linear code
               #C = 128,length = 5,and type 4^2,2^3 with length = gamma+delta";
G := Matrix(Z4, [[2,2,2,2,2],
		         [0,2,0,0,0],
                 [0,0,2,0,0],
                 [0,0,0,1,0],
                 [0,0,0,0,1]]);
C := LinearCode(G);
n := Length(C);
delta,gamma := Z4Type(C);
deltaDual := n-gamma-delta;

matrixHs := Matrix(Z4, [[0,0,2,0,0],
                        [0,0,0,2,0],
                        [0,0,0,0,2]]);
expectedOutputDualCs := LinearCode(matrixHs);
expectedOutputDualC := Dual(C);

expectedSpanCodeZ4 := C;
_,expectedSpanCodeZ2 := HasLinearGrayMapImage(expectedSpanCodeZ4);
                                             
expectedKernelCodeZ4 := C;
_,expectedKernelCodeZ2 := HasLinearGrayMapImage(expectedKernelCodeZ4);

// Test StandardFormInverted, MinRowsGeneratorMatrix functions
OutputMatrixGs, p := StandardFormInfoInverted(C);
OutputCs, f := StandardFormInverted(C);
OutputMatrixG := MinRowsGeneratorMatrix(C); 
assert Submatrix(OutputMatrixGs, gamma+1, n-delta+1, delta, delta) eq IdentityMatrix(Z4,delta);
assert Submatrix(OutputMatrixGs, 1, n-delta-gamma+1, gamma, gamma) eq 2*IdentityMatrix(Z4,gamma); 
assert OutputCs eq LinearCode(OutputMatrixGs);
assert OutputCs eq C^p;
assert OutputCs eq LinearCode(Matrix([f(OutputMatrixG[i]) : i in [1..gamma+delta]]));
assert C eq LinearCode(OutputMatrixG);
assert Nrows(OutputMatrixG) eq gamma+delta;
assert 2*RowSubmatrix(OutputMatrixG, gamma) eq ZeroMatrix(Z4, gamma, n);

// Test StandardFormDual, DualZ4 and MinRowsParityCheckMatrix functions
OutputMatrixHs, p := StandardFormDualInfo(C);
OutputDualCs, f := StandardFormDual(C);
OutputMatrixH := MinRowsParityCheckMatrix(C);
OutputDualC := DualZ4(C);
assert Submatrix(OutputMatrixHs, 1, delta+gamma+1, deltaDual, deltaDual) eq IdentityMatrix(Z4, deltaDual);
assert Submatrix(OutputMatrixHs, deltaDual+1, delta+1, gamma, gamma) eq 2*IdentityMatrix(Z4, gamma); 
assert OutputMatrixHs eq OutputMatrixH^p;
assert OutputMatrixHs eq Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]); 
assert OutputDualCs eq LinearCode(OutputMatrixHs);
assert OutputDualCs eq expectedOutputDualC^p; 
assert OutputDualCs eq LinearCode(Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]));
assert OutputDualC eq LinearCode(OutputMatrixH);
assert Nrows(OutputMatrixH) eq gamma+deltaDual;
assert 2*RowSubmatrix(OutputMatrixH, gamma) eq ZeroMatrix(Z4, gamma, n);
assert OutputDualC eq expectedOutputDualC;
assert OutputDualCs eq expectedOutputDualCs;

// Test SpanZ2CodeZ4, DimensionOfSpanZ2, KernelZ2CodeZ4, DimensionOfKernelZ2
OutputSpanCodeZ4, OutputSpanCodeZ2 := SpanZ2CodeZ4(C);
OutputKernelCodeZ4, OutputKernelCodeZ2 := KernelZ2CodeZ4(C);
OutputDimensionSpan := DimensionOfSpanZ2(C);
OutputDimensionKernel := DimensionOfKernelZ2(C);
assert OutputSpanCodeZ4 eq expectedSpanCodeZ4;
assert OutputSpanCodeZ2 eq expectedSpanCodeZ2;
assert OutputKernelCodeZ4 eq expectedKernelCodeZ4;
assert OutputKernelCodeZ2 eq expectedKernelCodeZ2;
assert OutputDimensionSpan eq Dimension(expectedSpanCodeZ2);
assert OutputDimensionKernel eq Dimension(expectedKernelCodeZ2);

/****************************************************************/
print "test 8: Z4-linear Reed-Muller code 
               #C = 536870912, length = 64, gamma = 3, delta = 13";
C := ReedMullerCodeRMZ4(2, 2, 5);
n := Length(C);
delta,gamma := Z4Type(C);
deltaDual := n-gamma-delta;

matrixHs := Matrix(Z4, [[0,1,2,2,1,0,0,1,0,1,0,0,0,0,0,0],
                        [3,2,3,0,0,1,2,0,0,0,1,0,0,0,0,0],
                        [0,1,2,0,0,2,1,1,0,0,0,1,0,0,0,0],
                        [3,2,3,1,2,0,0,0,0,0,0,0,1,0,0,0],
                        [1,1,0,3,3,3,3,0,1,0,0,0,0,1,0,0],
                        [3,0,3,0,2,0,2,0,1,0,0,0,0,0,1,0],
                        [3,0,2,1,1,1,1,1,1,0,0,0,0,0,0,1],
                        [2,2,2,0,0,0,0,2,0,0,0,0,0,0,0,0],
                        [2,0,0,2,0,2,0,0,2,0,0,0,0,0,0,0]]);
expectedOutputDualCs := LinearCode(matrixHs);
expectedOutputDualC := Dual(C);

expectedSpanCodeZ4 := LinearCode(Matrix(Z4, [[1,0,0,1,0,0,1,1,0,0,1,1,1,0,0,1],
                                             [0,1,0,1,0,0,0,0,0,0,0,0,0,1,0,1],
                                             [0,0,1,1,0,0,1,1,0,0,1,1,0,0,1,1],
                                             [0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,2],
                                             [0,0,0,0,1,0,1,0,0,0,0,0,1,0,1,0],
                                             [0,0,0,0,0,1,0,1,0,0,0,0,0,1,0,1],
                                             [0,0,0,0,0,0,2,0,0,0,0,0,0,0,2,0],
                                             [0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,2],
                                             [0,0,0,0,0,0,0,0,1,0,1,0,1,0,1,0],
                                             [0,0,0,0,0,0,0,0,0,1,0,1,0,1,0,1],
                                             [0,0,0,0,0,0,0,0,0,0,2,0,0,0,2,0],
                                             [0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,2],
                                             [0,0,0,0,0,0,0,0,0,0,0,0,2,0,2,0],
                                             [0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,2]]));
_,expectedSpanCodeZ2 := HasLinearGrayMapImage(expectedSpanCodeZ4);
                                             
expectedKernelCodeZ4 := LinearCode(Matrix(Z4, [[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],
                                 [0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1],
                                 [0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0],
                                 [0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2],
                                 [0,0,0,0,2,0,2,0,0,0,0,0,0,2,0,2],
                                 [0,0,0,0,0,2,0,2,0,0,0,0,0,2,0,2],
                                 [0,0,0,0,0,0,0,0,2,0,2,0,0,2,0,2],
                                 [0,0,0,0,0,0,0,0,0,2,0,2,0,2,0,2],
                                 [0,0,0,0,0,0,0,0,0,0,0,0,2,2,2,2]]));
_,expectedKernelCodeZ2 := HasLinearGrayMapImage(expectedKernelCodeZ4);

// Test StandardFormInverted, MinRowsGeneratorMatrix functions
OutputMatrixGs, p := StandardFormInfoInverted(C);
OutputCs, f := StandardFormInverted(C);
OutputMatrixG := MinRowsGeneratorMatrix(C); 
assert Submatrix(OutputMatrixGs, gamma+1, n-delta+1, delta, delta) eq IdentityMatrix(Z4,delta);
assert Submatrix(OutputMatrixGs, 1, n-delta-gamma+1, gamma, gamma) eq 2*IdentityMatrix(Z4,gamma); 
assert OutputCs eq LinearCode(OutputMatrixGs);
assert OutputCs eq C^p;
assert OutputCs eq LinearCode(Matrix([f(OutputMatrixG[i]) : i in [1..gamma+delta]]));
assert C eq LinearCode(OutputMatrixG);
assert Nrows(OutputMatrixG) eq gamma+delta;
assert 2*RowSubmatrix(OutputMatrixG, gamma) eq ZeroMatrix(Z4, gamma, n);

// Test StandardFormDual, DualZ4 and MinRowsParityCheckMatrix functions
OutputMatrixHs, p := StandardFormDualInfo(C);
OutputDualCs, f := StandardFormDual(C);
OutputMatrixH := MinRowsParityCheckMatrix(C);
OutputDualC := DualZ4(C);
assert Submatrix(OutputMatrixHs, 1, delta+gamma+1, deltaDual, deltaDual) eq IdentityMatrix(Z4, deltaDual);
assert Submatrix(OutputMatrixHs, deltaDual+1, delta+1, gamma, gamma) eq 2*IdentityMatrix(Z4, gamma); 
assert OutputDualCs eq LinearCode(OutputMatrixHs);
assert OutputDualCs eq expectedOutputDualC^p; 
assert OutputDualCs eq LinearCode(Matrix([f(OutputMatrixH[i]) : i in [1..Nrows(OutputMatrixH)]]));
assert OutputDualC eq LinearCode(OutputMatrixH);
assert Nrows(OutputMatrixH) eq gamma+deltaDual;
assert 2*RowSubmatrix(OutputMatrixH, gamma) eq ZeroMatrix(Z4, gamma, n);
assert OutputDualC eq expectedOutputDualC;
assert OutputDualCs eq expectedOutputDualCs;

// Test SpanZ2CodeZ4, DimensionOfSpanZ2, KernelZ2CodeZ4, DimensionOfKernelZ2
OutputSpanCodeZ4, OutputSpanCodeZ2 := SpanZ2CodeZ4(C);
OutputKernelCodeZ4, OutputKernelCodeZ2 := KernelZ2CodeZ4(C);
OutputDimensionSpan := DimensionOfSpanZ2(C);
OutputDimensionKernel := DimensionOfKernelZ2(C);
assert OutputSpanCodeZ4 eq expectedSpanCodeZ4;
assert OutputSpanCodeZ2 eq expectedSpanCodeZ2;
assert OutputKernelCodeZ4 eq expectedKernelCodeZ4;
assert OutputKernelCodeZ2 eq expectedKernelCodeZ2;
assert OutputDimensionSpan eq Dimension(expectedSpanCodeZ2);
assert OutputDimensionKernel eq Dimension(expectedKernelCodeZ2);
